create or replace package body tbicds.PCK_REPORTS_STAT is
/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
  /*
  Create a new export request.
  */
  procedure CreateExportRequestRS(pi_vSessionID         in varchar2,
                                  pi_vSessionClientIP   in varchar2,
                                  pi_nUserID            in number,
                                  pi_vExportRequestName in varchar2,
                                  po_nStatusCode        out number,
                                  po_vStatusComment     out varchar2,
                                  rs                    out RetRefCursor)
  is

    v_seqRequestID   number;
    v_nRptNameExists number := 0;

  begin

    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';

    begin
      select count(*)
        into v_nRptNameExists
        from exp_statistical_rpt t
       where t.fx_user_id = pi_nUserID
         and t.report_name = pi_vExportRequestName;
    end;

    if v_nRptNameExists = 0 then
      --get a new request id
      select seqexpstatisticalrpt.nextval into v_seqRequestID from dual;

      --create the export request record...
      insert into exp_statistical_rpt(request_id,fx_user_id,report_name,status)
      values(
        v_seqRequestID,
        pi_nUserID,
        pi_vExportRequestName,
        0 --new request pending for definition
      );
      commit;
      
      open rs for
        select v_seqRequestID as REQUEST_ID from dual;
    else
      --raise exception
      po_nStatusCode    := 2;
      po_vStatusComment := 'Export request already exists.';
    end if;

  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_REPORTS_STAT.CreateExportRequest(): ' || sqlErrm;
  end;

  /*
  Get a list of the request of this user.
  */
  procedure GetMyRequestRS(pi_vSessionID       in varchar2,
                           pi_vSessionClientIP in varchar2,
                           pi_nUserID          in number,
                           po_nStatusCode      out number,
                           po_vStatusComment   out varchar2,
                           rs                  out RetRefCursor)
  is

  begin

    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';

    open rs for
      select
        t.request_id,
        t.report_name,
        t.status as status_code,
        case t.status
          when -1 then 'Error'
          when  0 then 'New (Pending for Execution)'
          when  1 then 'In Progress'
          when  2 then 'Ready for Download'
          when  3 then 'Canceled' 
        end as status_description,
        t.start_date as str_date,
        t.end_date as end_date
      from exp_statistical_rpt t
      where t.fx_user_id = pi_nUserID
      order by t.request_id desc;

  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_REPORTS_STAT.GetMyRequestRS(): ' || sqlErrm;
  end;

  /*
  Rename export request
  */
  procedure RenameExportRequest(pi_vSessionID            in varchar2,
                                pi_vSessionClientIP      in varchar2,
                                pi_nUserID               in number,
                                pi_nRequestID            in number,
                                pi_vNewExportRequestName in varchar2,
                                po_nStatusCode           out number,
                                po_vStatusComment        out varchar2)
  is

    v_nRptNameExists number := 0;
    v_nTargetExists  number := 0;

  begin

    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';

    --val 1: new export request name already exists?
    begin
      select count(*)
        into v_nRptNameExists
        from exp_statistical_rpt t
       where t.fx_user_id = pi_nUserID
         and t.report_name = pi_vNewExportRequestName;
    end;

    if v_nRptNameExists = 0 then
      --val 2: target request to be rename, exists?
      begin
        select count(*)
          into v_nTargetExists
          from exp_statistical_rpt t
         where t.fx_user_id = pi_nUserID
           and t.request_id = pi_nRequestID;
        end;

        if v_nTargetExists = 0 then
          --raise exception
          po_nStatusCode    := 3;
          po_vStatusComment := 'Export request not found.';

        else
          --rename
          update exp_statistical_rpt set
            report_name = pi_vNewExportRequestName
          where fx_user_id = pi_nUserID
            and request_id = pi_nRequestID;
          commit;
        end if;
    else
      --raise exception
      po_nStatusCode    := 2;
      po_vStatusComment := 'Export request already exists.';
    end if;

  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_REPORTS_STAT.RenameExportRequest(): ' || sqlErrm;
  end;

  /*
  Update export request
  */
  procedure UpdateExportRequest(pi_vSessionID       in varchar2,
                                pi_vSessionClientIP in varchar2,
                                pi_nUserID          in number,
                                pi_nRequestID       in number,
                                pi_vFilters         in varchar2,
                                pi_vDataFields      in varchar2,
                                po_nStatusCode      out number,
                                po_vStatusComment   out varchar2)
  is

    v_nTargetExists  number := 0;

  begin

    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';

    --val 1: target request exists?
    begin
      select count(*)
        into v_nTargetExists
        from exp_statistical_rpt t
       where t.fx_user_id = pi_nUserID
         and t.request_id = pi_nRequestID
         and t.status = 0; --Only New request can be updated
    end;

    if v_nTargetExists = 1 then
      --update filters and datafields
        update exp_statistical_rpt set
          filters = pi_vFilters,
          data_included = pi_vDataFields
        where fx_user_id = pi_nUserID
          and request_id = pi_nRequestID
          and status = 0; --Only New request can be updated
        commit;
    else
      --raise exception
      if v_nTargetExists = 0 then
        po_nStatusCode    := 2;
        po_vStatusComment := 'Request not found.';
      else
        po_nStatusCode    := 3;
        po_vStatusComment := 'Multiple request found with the same request ID.'; --this won't happend because it will be a primary key violation.
      end if;
    end if;

  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_REPORTS_STAT.UpdateExportRequest(): ' || sqlErrm;
  end;

  /*
  Get export request recordset
  */
  procedure GetExportRequestRS(pi_vSessionID       in varchar2,
                               pi_vSessionClientIP in varchar2,
                               pi_nUserID          in number,
                               pi_nRequestID       in number,
                               po_nStatusCode      out number,
                               po_vStatusComment   out varchar2,
                               rs                  out RetRefCursor)
  is

    v_nTargetExists  number := 0;

  begin

    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';

    --val 1: target request exists?
    begin
      select count(*)
        into v_nTargetExists
        from exp_statistical_rpt t
       where t.fx_user_id = pi_nUserID
         and t.request_id = pi_nRequestID;
    end;

    if v_nTargetExists = 1 then
      open rs for
        select t.report_name,
               t.filters,
               t.data_included
          from exp_statistical_rpt t
         where t.fx_user_id = pi_nUserID
           and t.request_id = pi_nRequestID;
    else
      --raise exception
      if v_nTargetExists = 0 then
        po_nStatusCode    := 2;
        po_vStatusComment := 'Request not found.';
      else
        po_nStatusCode    := 3;
        po_vStatusComment := 'Multiple request found with the same request ID.'; --this won't happend because it will be a primary key violation.
      end if;
    end if;

  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_REPORTS_STAT.GetExportRequestRS(): ' || sqlErrm;
  end;

  /*
  Save export request
  */
  procedure SaveAsExportRequest(pi_vSessionID         in varchar2,
                                pi_vSessionClientIP   in varchar2,
                                pi_nUserID            in number,
                                pi_vExportRequestName in varchar2,
                                pi_vFilters         in varchar2,
                                pi_vDataFields      in varchar2,
                                po_nStatusCode        out number,
                                po_vStatusComment     out varchar2)
  is

    v_seqRequestID   number;
    v_nRptNameExists number := 0;

  begin

    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';

    --val 1: target request exists?
    begin
      select count(*)
        into v_nRptNameExists
        from exp_statistical_rpt t
       where t.fx_user_id = pi_nUserID
         and t.report_name = pi_vExportRequestName;
    end;

    if v_nRptNameExists = 0 then
      --get a new request id
      select seqexpstatisticalrpt.nextval into v_seqRequestID from dual;

      --create the export request record...
      insert into exp_statistical_rpt(request_id,fx_user_id,report_name,status,filters,data_included)
      values(
        v_seqRequestID,
        pi_nUserID,
        pi_vExportRequestName,
        0, --new request pending for definition
        pi_vFilters,
        pi_vDataFields
      );
      commit;
    else
      --raise exception
      po_nStatusCode    := 2;
      po_vStatusComment := 'Export request already exists.';
    end if;

  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_REPORTS_STAT.SaveAsExportRequest(): ' || sqlErrm;
  end;
  
  /*
  Submit export request
  */
  procedure SubmitExportRequest(pi_vSessionID       in varchar2,
                                pi_vSessionClientIP in varchar2,
                                pi_nUserID          in number,
                                pi_vKey             in varchar2,
                                pi_nRequestID       in number,
                                pi_vFilters         in varchar2,
                                pi_vDataFields      in varchar2,
                                po_nStatusCode      out number,
                                po_vStatusComment   out varchar2)
  is

    v_nTargetExists  number := 0;

  begin

    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';

    --val 1: target request exists?
    begin
      select count(*)
        into v_nTargetExists
        from exp_statistical_rpt t
       where t.fx_user_id = pi_nUserID
         and t.request_id = pi_nRequestID
         and t.status = 0; --Only New request can be updated
    end;

    if v_nTargetExists = 1 then
      --update filters, datafields, status, and start_date
        update exp_statistical_rpt set
          filters = pi_vFilters,
          data_included = pi_vDataFields,
          status = 1, --In Progress
          start_date = sysdate
        where fx_user_id = pi_nUserID
          and request_id = pi_nRequestID
          and status = 0; --Only New request can be updated
        commit;
        
        --CALL CSV BUILDER PROCESS
        PCK_UTL_EXPORT_DATA.runExport (
          pi_vSessionID,
          pi_vSessionClientIP,
          pi_nUserID,
          pi_nRequestID,
          pi_vKey,
          po_nStatusCode,
          po_vStatusComment
        );

    else
      --raise exception
      if v_nTargetExists = 0 then
        po_nStatusCode    := 2;
        po_vStatusComment := 'Request not found.';
      else
        po_nStatusCode    := 3;
        po_vStatusComment := 'Multiple request found with the same request ID.'; --this won't happend because it will be a primary key violation.
      end if;
    end if;

  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_REPORTS_STAT.SubmitExportRequest(): ' || sqlErrm;
  end;
  
  /*
  Cancel export request
  */
  procedure CancelExportRequest(pi_vSessionID       in varchar2,
                                pi_vSessionClientIP in varchar2,
                                pi_nUserID          in number,
                                pi_nRequestID       in number,
                                po_nStatusCode      out number,
                                po_vStatusComment   out varchar2)
  is

    v_nTargetExists  number := 0;

  begin

    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';

    --val 1: target request exists?
    begin
      select count(*)
        into v_nTargetExists
        from exp_statistical_rpt t
       where t.fx_user_id = pi_nUserID
         and t.request_id = pi_nRequestID
         and t.status = 1; --Only In Progress request can be cancel
    end;

    if v_nTargetExists = 1 then
      --update status and end_date
        update exp_statistical_rpt set
          status = 3, --Canceled
          end_date = sysdate
        where fx_user_id = pi_nUserID
          and request_id = pi_nRequestID
          and status = 1; --Only In Progress request can be cancel
        commit;
        
        --STOP CSV BUILDER PROCESS???
        PCK_UTL_EXPORT_DATA.cancelExport (
          pi_vSessionID,
          pi_vSessionClientIP,
          pi_nUserID,
          pi_nRequestID,
          po_nStatusCode,
          po_vStatusComment
        );

    else
      --raise exception
      if v_nTargetExists = 0 then
        po_nStatusCode    := 2;
        po_vStatusComment := 'Request not found.';
      else
        po_nStatusCode    := 3;
        po_vStatusComment := 'Multiple request found with the same request ID.'; --this won't happend because it will be a primary key violation.
      end if;
    end if;

  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_REPORTS_STAT.CancelExportRequest(): ' || sqlErrm;
  end;
  
  /*
  Get export request CSV
  */
  procedure GetExportRequestCSVFileRS(pi_vSessionID       in varchar2,
                                      pi_vSessionClientIP in varchar2,
                                      pi_nUserID          in number,
                                      pi_nRequestID       in number,
                                      po_nStatusCode      out number,
                                      po_vStatusComment   out varchar2,
                                      rs                  out RetRefCursor)
  is

    v_nTargetExists  number := 0;

  begin

    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';

    --val 1: target request exists?
    begin
      select count(*)
        into v_nTargetExists
        from exp_statistical_rpt t
       where t.fx_user_id = pi_nUserID
         and t.request_id = pi_nRequestID
         and t.status = 2;
    end;

    if v_nTargetExists = 1 then
      open rs for
        select t.report_name,
               t.file_data
          from exp_statistical_rpt t
         where t.fx_user_id = pi_nUserID
           and t.request_id = pi_nRequestID
           and t.status = 2;
           
      --AUDIT
      insert into exp_statistical_rpt_audit(request_id,fx_user_id,request_download_date) 
      values(pi_nRequestID,
             pi_nUserID,
             sysdate);
      commit;
      
    else
      --raise exception
      if v_nTargetExists = 0 then
        po_nStatusCode    := 2;
        po_vStatusComment := 'Request not found.';
      else
        po_nStatusCode    := 3;
        po_vStatusComment := 'Multiple request found with the same request ID.'; --this won't happend because it will be a primary key violation.
      end if;
    end if;

  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_REPORTS_STAT.GetExportRequestCSVFileRS(): ' || sqlErrm;
  end;
  
  /*
  Submit new export request
  */
  procedure SubmitNewExportRequest(pi_vSessionID         in varchar2,
                                   pi_vSessionClientIP   in varchar2,
                                   pi_nUserID            in number,
                                   pi_vKey               in varchar2,
                                   pi_vExportRequestName in varchar2,
                                   pi_vFilters           in varchar2,
                                   pi_vDataFields        in varchar2,
                                   po_nStatusCode        out number,
                                   po_vStatusComment     out varchar2)
  is

    v_seqRequestID   number;
    v_nRptNameExists number := 0;

  begin

    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';

    begin
      select count(*)
        into v_nRptNameExists
        from exp_statistical_rpt t
       where t.fx_user_id = pi_nUserID
         and t.report_name = pi_vExportRequestName;
    end;

    if v_nRptNameExists = 0 then
      --get a new request id
      select seqexpstatisticalrpt.nextval into v_seqRequestID from dual;

      --create the export request record...
      insert into exp_statistical_rpt(request_id,fx_user_id,report_name,filters,data_included,status,start_date)
      values(
        v_seqRequestID,
        pi_nUserID,
        pi_vExportRequestName,
        pi_vFilters,
        pi_vDataFields,
        1, --In Progress
        sysdate
      );
      commit;
      
      --CALL CSV BUILDER PROCESS
      PCK_UTL_EXPORT_DATA.runExport (
        pi_vSessionID,
        pi_vSessionClientIP,
        pi_nUserID,
        v_seqRequestID,
        pi_vKey,
        po_nStatusCode,
        po_vStatusComment
      );
      
    else
      --raise exception
      po_nStatusCode    := 2;
      po_vStatusComment := 'Export request already exists.';
    end if;

  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_REPORTS_STAT.SubmitNewExportRequest(): ' || sqlErrm;
  end;

end PCK_REPORTS_STAT;
/

